<?php
// Enable PHP error logging
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/php-error.log');
error_reporting(E_ALL);

header('Content-Type: application/json');
require_once 'db.php';

// Read raw POST data

$input = file_get_contents('php://input');
$data = json_decode($input, true);

// Get api_key from Bearer header only (do not expect in payload)
$api_key = null;
$headers = getallheaders();
if (isset($headers['Authorization']) && preg_match('/Bearer\s(\S+)/', $headers['Authorization'], $matches)) {
    $api_key = $matches[1];
}

if (
    !$data ||
    !isset($data['printers']) ||
    !$api_key
) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Invalid payload']);
    exit;
}

// Verify api_key
$user_stmt = $conn->prepare("SELECT id, webhook_url FROM users WHERE api_key=?");
$user_stmt->bind_param("s", $api_key);
$user_stmt->execute();
$user_stmt->bind_result($user_id, $webhook_url);
$user_stmt->fetch();
$user_stmt->close();
if (!$user_id) {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => 'Invalid API key']);
    exit;
}

$client_id = $data['client_id'];
$printers = $data['printers'];
$added = 0; $updated = 0;

foreach ($printers as $printer) {
    $unique_id = $printer['id'];
    $name = $printer['name'];
    $status = $printer['status'];
    $enabled = $printer['enabled'];

    // Check if printer exists
    $stmt = $conn->prepare("SELECT id FROM tbl_printers WHERE unique_id=?");
    $stmt->bind_param("s", $unique_id);
    $stmt->execute();
    $stmt->store_result();

    if ($stmt->num_rows > 0) {
        // Update printer
        $update = $conn->prepare("UPDATE tbl_printers SET name=?, status=?, client_id=?, enabled=?, user_id=?, last_updated=NOW() WHERE unique_id=?");
        $update->bind_param("sssiis", $name, $status, $client_id, $enabled, $user_id, $unique_id);
        if ($update->execute()) $updated++;
    } else {
        // Insert printer (minimal fields)
        $insert = $conn->prepare("INSERT INTO tbl_printers (unique_id, name, type, characters_per_line, print_format, status, client_id, enabled, user_id) VALUES (?, ?, '', 32, '', ?, ?, ?, ?)");
        $insert->bind_param("ssssii", $unique_id, $name, $status, $client_id, $enabled, $user_id);
        if ($insert->execute()) $added++;
    }
}

http_response_code(200);
$response = [
    'success' => true,
    'added' => $added,
    'updated' => $updated
];
echo json_encode($response);

// Forward webhook to user's webhook_url if set and not empty
if (!empty($webhook_url)) {
    $ch = curl_init($webhook_url);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $input);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Authorization: Bearer ' . $api_key
    ]);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // Ignore SSL
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false); // Ignore SSL
    curl_exec($ch);
    curl_close($ch);
}

exit;
?>
</html>
