<?php
header('Content-Type: application/json');
include 'db.php';

// Helper: get user id from api_key
function get_user_id($conn, $api_key) {
    $stmt = $conn->prepare("SELECT id FROM users WHERE api_key=?");
    $stmt->bind_param("s", $api_key);
    $stmt->execute();
    $stmt->bind_result($user_id);
    if ($stmt->fetch()) return $user_id;
    return false;
}

// Support both JSON, POST, and GET
$input = file_get_contents('php://input');
$json = json_decode($input, true);
$post = $_POST;
$get = $_GET;

// Merge JSON, POST, and GET. Priority: POST > JSON > GET
$data = array_merge($get, is_array($json) ? $json : [], $post);

// Get api_key from data or Bearer header
$api_key = $data['api_key'] ?? null;
if (!$api_key) {
    $headers = getallheaders();
    if (isset($headers['Authorization']) && preg_match('/Bearer\s(\S+)/', $headers['Authorization'], $matches)) {
        $api_key = $matches[1];
    }
}

$action = $data['action'] ?? '';

if (!$api_key) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Missing api_key']);
    exit;
}

$user_id = get_user_id($conn, $api_key);
if (!$user_id) {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => 'Invalid api_key']);
    exit;
}

if ($action === 'get_printers') {
    // Get all printers for this user (assuming printers are linked to user_id)
    $printers = [];
    $stmt = $conn->prepare("SELECT id, unique_id, name, type, characters_per_line, print_format, status, time_added, last_updated FROM tbl_printers WHERE user_id=?");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) $printers[] = $row;
    echo json_encode(['success' => true, 'printers' => $printers]);
    exit;
}

if ($action === 'send_printjob') {

    
        // Read data from merged $data
        $printer_id = $data['printer_id'] ?? '';
        $type = $data['type'] ?? '';
        $content = $data['content'] ?? '';

        if (!$api_key || !$printer_id || !$type || !$content) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'Missing parameters']);
            exit;
        }

    // Verify api_key and get user_id
        $user_stmt = $conn->prepare("SELECT id FROM users WHERE api_key=?");
        $user_stmt->bind_param("s", $api_key);
        $user_stmt->execute();
        $user_stmt->bind_result($user_id);
        if (!$user_stmt->fetch()) {
            http_response_code(403);
            echo json_encode(['success' => false, 'message' => 'Invalid api_key']);
            exit;
        }
        $user_stmt->close();

        // Verify printer belongs to user (tbl_printers must have user_id column)
        $printer_stmt = $conn->prepare("SELECT id FROM tbl_printers WHERE unique_id=? AND user_id=?");
        $printer_stmt->bind_param("si", $printer_id, $user_id);
        $printer_stmt->execute();
        $printer_stmt->store_result();
        if ($printer_stmt->num_rows == 0) {
            http_response_code(403);
            echo json_encode(['success' => false, 'message' => 'Printer does not belong to user']);
            exit;
        }
        $printer_stmt->close();

        $content = base64_encode($content);
        // Save job
        $job_data = json_encode(['printer_id' => $printer_id, 'type' => $type, 'content' => $content]);
        $insert_job = $conn->prepare("INSERT INTO tbl_jobs (printer_id, user_id, job_data, status, time_added) VALUES (?, ?, ?, 'done', NOW())");
        $insert_job->bind_param("sis", $printer_id, $user_id, $job_data);
        $insert_job->execute();
        $insert_job->close();

        $payload = [
            'job_id' => uniqid('job_'),          // or pass an actual job_id from DB
            'printer_id' => $printer_id,
            'api_key' => $api_key,
            'content' => $content,              // previously `content`
            'type' => $type              // previously `type`
        ];
        
        // Send POST request
        $ch = curl_init($socket_http_url_send_job);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // Don't verify SSL certificate
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false); // Don't verify hostname
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json'
        ]);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if (curl_errno($ch)) {
            // echo 'cURL error: ' . curl_error($ch);
        } else {
            // echo "Response Code: $httpCode\n";
            // echo "Response Body:\n$response";
        }

        curl_close($ch);

        if ($httpCode === 200) {
            echo json_encode(['success' => true, 'message' => 'Print job saved and forwarded']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Failed to forward print job']);
        }
        exit;
    } else {

    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Invalid action']);
}
?>
